from sqlalchemy import Column, Integer, String, Table, ForeignKey
from sqlalchemy.orm import Mapped, relationship, mapped_column
from typing import List
from datetime import datetime
from portfolio import Base
from sqlalchemy import select
from flask_login import UserMixin

tools = Table('tools',
Base.metadata,
Column("project_id", ForeignKey("Project.id"), primary_key=True),
Column("framework_id", ForeignKey("Framework.id"), primary_key=True)
)

class EmploymentORM(Base):
    __tablename__ = "Employment"

    id: Mapped[int] = mapped_column(primary_key=True)
    role: Mapped[str] = mapped_column(String)
    dates: Mapped[str] = mapped_column(String)
    duties_1: Mapped[str] = mapped_column(String)
    duties_2: Mapped[str] = mapped_column(String)
    duties_3: Mapped[str] = mapped_column(String)

    location_id: Mapped[int] = mapped_column(ForeignKey("Location.id"))
    location: Mapped["LocationORM"] = relationship("LocationORM", back_populates="employments")

class LocationORM(Base):
    __tablename__ = "Location"

    id: Mapped[int] = mapped_column(primary_key=True)
    location_name: Mapped[str] = mapped_column(String)
    city: Mapped[str] = mapped_column(String)

    employments: Mapped[list["EmploymentORM"]] = relationship("EmploymentORM", back_populates="location")
    qualifications: Mapped[list["QualificationORM"]] = relationship("QualificationORM", back_populates="location")

class QualificationORM(Base):
    __tablename__ = "Qualification"

    id: Mapped[int] = mapped_column(primary_key=True)
    qual_name: Mapped[str] = mapped_column(String)
    awarding_body: Mapped[str] = mapped_column(String)
    dates: Mapped[str] = mapped_column(String)
    grade: Mapped[str] = mapped_column(String)

    location_id: Mapped[int] = mapped_column(ForeignKey("Location.id"))
    location: Mapped["LocationORM"] = relationship("LocationORM", back_populates="qualifications")

class FrameworkORM(Base):
    __tablename__ = "Framework"
    id = Column(Integer, primary_key=True)
    framework_name = Column(String)
    icon = Column(String)
    link = Column(String)

class ProjectORM(Base):
    __tablename__ = "Project"
    Base.metadata
    id = Column(Integer, primary_key=True)
    project_name = Column(String)
    link = Column(String)
    slug = Column(String)

    tools : Mapped[List["FrameworkORM"]] = relationship(secondary="tools")

class BlogpostORM(Base):
    __tablename__ = "Blogposts"
    id : Mapped[int] = mapped_column(primary_key=True)
    author = Column(String)
    slug = Column(String)
    content = Column(String)
    timestamp = Column(Integer)
    comments: Mapped[list["CommentORM"]] = relationship("CommentORM", back_populates="post")

class CommentORM(Base):
    __tablename__ = "Comments"
    id : Mapped[int] = mapped_column(primary_key=True)
    author = Column(String)
    comment = Column(String)
    timestamp = Column(Integer)
    post_id: Mapped[int] = mapped_column(ForeignKey("Blogposts.id"))
    post : Mapped["BlogpostORM"] = relationship("BlogpostORM", back_populates="comments")

class UserORM(Base, UserMixin):
    __tablename__ = "Users"

    id: Mapped[int] = mapped_column(primary_key=True)
    name: Mapped[str] = mapped_column(String, unique=True, nullable=False)
    email: Mapped[str] = mapped_column(String, nullable=False, unique=True)
    password: Mapped[str] = mapped_column(String, nullable=False)

class BlogpostDAO():
    def allBlogposts(db):
        stm = select(BlogpostORM)
        blogposts = db.session.scalars(stm).all()
        return blogposts

    def blogpostByID():
        pass

    def blogpostBySlug(db, slug):
        stm = select(BlogpostORM).where(BlogpostORM.slug == slug)
        blogpost = db.session.scalar(stm)
        return blogpost

    def createBlogpost(db, post):
        db.session.add(post)
        db.session.commit()
        return post

    def deleteBlogpostByID(db, id):
        post_to_delete = BlogpostORM.query.get(id)
        db.session.delete(post_to_delete)
        db.session.commit()

class CommentDAO():
    def allCommentsByPostID(id):
        stm = select(CommentORM).where(CommentORM.post_id == id )
        comments = db.session.scalars(stm).all()
        return comments

    def createComment(db, new_comment):
        try:
            db.session.add(new_comment)
            db.session.commit()
        except Exception as e:
            db.session.rollback()
            print(f"Error committing comment: {e}")

    def deleteCommentByID(db, id):
        post_to_delete = CommentORM.query.get(id)
        db.session.delete(post_to_delete)
        db.session.commit()

class EmploymentDAO():
    def allEmployment(db):
        stm = select(EmploymentORM)
        employment = db.session.scalars(stm).all()

        return employment

class QualificationDAO():
    def allQualifications(db):
        stm = select(QualificationORM)
        qualifications = db.session.scalars(stm).all()

        return qualifications

class ProjectDAO():
    def allProjects(db):
        stm = select(ProjectORM)
        projects = db.session.scalars(stm).all()

        return projects

class UserDAO():
    def addNewUser(db, new_user):
        try:
            db.session.add(new_user)
            db.session.commit()
        except Exception as e:
            db.session.rollback()
            print(f"Error creating user: {e}")
        return new_user